/*
 * Decompiled with CFR 0.152.
 */
package io.nuls.account.service.impl;

import com.fasterxml.jackson.core.JsonProcessingException;
import io.nuls.account.constant.AccountErrorCode;
import io.nuls.account.model.bo.Account;
import io.nuls.account.model.bo.AccountKeyStore;
import io.nuls.account.model.bo.Chain;
import io.nuls.account.model.po.AccountPO;
import io.nuls.account.rpc.call.EventCall;
import io.nuls.account.service.AccountCacheService;
import io.nuls.account.service.AccountKeyStoreService;
import io.nuls.account.service.AccountService;
import io.nuls.account.service.AliasService;
import io.nuls.account.storage.AccountForTransferOnContractCallStorageService;
import io.nuls.account.storage.AccountStorageService;
import io.nuls.account.util.AccountTool;
import io.nuls.account.util.LoggerUtil;
import io.nuls.account.util.Preconditions;
import io.nuls.base.basic.AddressTool;
import io.nuls.base.data.Address;
import io.nuls.base.data.NulsSignData;
import io.nuls.base.signture.BlockSignature;
import io.nuls.base.signture.P2PHKSignature;
import io.nuls.base.signture.SignatureUtil;
import io.nuls.core.core.annotation.Autowired;
import io.nuls.core.core.annotation.Component;
import io.nuls.core.crypto.AESEncrypt;
import io.nuls.core.crypto.Base58;
import io.nuls.core.crypto.ECKey;
import io.nuls.core.crypto.HexUtil;
import io.nuls.core.exception.CryptoException;
import io.nuls.core.exception.NulsException;
import io.nuls.core.exception.NulsRuntimeException;
import io.nuls.core.model.FormatValidUtils;
import io.nuls.core.model.StringUtils;
import io.nuls.core.parse.JSONUtils;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

@Component
public class AccountServiceImpl
implements AccountService {
    private Lock locker = new ReentrantLock();
    @Autowired
    private AccountStorageService accountStorageService;
    @Autowired
    private AliasService aliasService;
    @Autowired
    private AccountKeyStoreService keyStoreService;
    @Autowired
    private AccountForTransferOnContractCallStorageService accountForTransferOnContractCallStorageService;
    private AccountCacheService accountCacheService = AccountCacheService.getInstance();

    @Override
    public List<Account> createAccount(Chain chain, int count, String password) {
        int chainId = chain.getChainId();
        if (chainId <= 0 || count <= 0 || count > 100 || StringUtils.isBlank((String)password)) {
            throw new NulsRuntimeException(AccountErrorCode.PARAMETER_ERROR);
        }
        if (!FormatValidUtils.validPassword((String)password)) {
            throw new NulsRuntimeException(AccountErrorCode.PASSWORD_FORMAT_WRONG);
        }
        this.locker.lock();
        try {
            ArrayList<Account> accounts = new ArrayList<Account>();
            ArrayList<AccountPO> accountPOs = new ArrayList<AccountPO>();
            for (int i = 0; i < count; ++i) {
                Account account = AccountTool.createAccount(chainId);
                account.encrypt(password);
                accounts.add(account);
                AccountPO po = new AccountPO(account);
                accountPOs.add(po);
            }
            boolean result = this.accountStorageService.saveAccountList(accountPOs);
            if (result) {
                for (Account account : accounts) {
                    this.accountCacheService.getLocalAccountMaps().put(account.getAddress().getBase58(), account);
                    this.keyStoreService.backupAccountToKeyStore(null, chainId, account.getAddress().getBase58(), password);
                    HashMap<String, Object> eventData = new HashMap<String, Object>();
                    eventData.put("address", account.getAddress().getBase58());
                    eventData.put("isEncrypted", account.isEncrypted());
                    EventCall.sendEvent("evt_ac_createAccount", JSONUtils.obj2json(eventData));
                }
            }
            ArrayList<Account> arrayList = accounts;
            return arrayList;
        }
        catch (NulsException e) {
            chain.getLogger().error(e);
            throw new NulsRuntimeException(e.getErrorCode());
        }
        catch (JsonProcessingException e) {
            chain.getLogger().error((Exception)((Object)e));
            throw new NulsRuntimeException(AccountErrorCode.PARSE_JSON_FAILD);
        }
        finally {
            this.locker.unlock();
        }
    }

    @Override
    public Account getAccount(int chainId, String address) {
        if (!AddressTool.validAddress((int)chainId, (String)address)) {
            throw new NulsRuntimeException(AccountErrorCode.ADDRESS_ERROR);
        }
        Account account = this.getAccountByAddress(chainId, address);
        return account;
    }

    @Override
    public boolean validationWhitelistForTransferOnContractCall(int chainId, String address) {
        if (!AddressTool.validAddress((int)chainId, (String)address)) {
            throw new NulsRuntimeException(AccountErrorCode.ADDRESS_ERROR);
        }
        return this.accountForTransferOnContractCallStorageService.exist(AddressTool.getAddress((String)address));
    }

    @Override
    public List<Account> getAccountList() {
        ArrayList<Account> list = new ArrayList<Account>();
        if (this.accountCacheService.getLocalAccountMaps().size() > 0) {
            Collection<Account> values = this.accountCacheService.getLocalAccountMaps().values();
            Iterator<Account> iterator = values.iterator();
            while (iterator.hasNext()) {
                list.add(iterator.next());
            }
        } else {
            List<AccountPO> poList = this.accountStorageService.getAccountList();
            if (null == poList || poList.isEmpty()) {
                return list;
            }
            for (AccountPO po : poList) {
                Account account = po.toAccount();
                list.add(account);
            }
            for (Account account : list) {
                this.accountCacheService.getLocalAccountMaps().put(account.getAddress().getBase58(), account);
            }
        }
        Collections.sort(list, (o1, o2) -> o2.getCreateTime().compareTo(o1.getCreateTime()));
        return list;
    }

    @Override
    public List<Account> getAccountListByChain(int chainId) {
        ArrayList<Account> accountList = new ArrayList<Account>();
        List<Account> accounts = this.getAccountList();
        accounts.stream().filter(acc -> acc.getChainId() == chainId).forEach(account -> accountList.add((Account)account));
        return accountList;
    }

    private Account getAccountByAddress(int chainId, String address) {
        if (!AddressTool.validAddress((int)chainId, (String)address)) {
            return null;
        }
        if (this.accountCacheService.getLocalAccountMaps() == null || this.accountCacheService.getLocalAccountMaps().size() == 0) {
            this.getAccountList();
        }
        if (this.accountCacheService.getLocalAccountMaps() != null) {
            return this.accountCacheService.getLocalAccountMaps().get(address);
        }
        return null;
    }

    @Override
    public boolean setPassword(int chainId, String address, String password) {
        if (!AddressTool.validAddress((int)chainId, (String)address)) {
            LoggerUtil.LOG.debug("the address is illegal,chainId:{},address:{}", new Object[]{chainId, address});
            throw new NulsRuntimeException(AccountErrorCode.ADDRESS_ERROR);
        }
        if (StringUtils.isBlank((String)password)) {
            LoggerUtil.LOG.debug("the password should't be null,chainId:{},address:{}", new Object[]{chainId, address});
            throw new NulsRuntimeException(AccountErrorCode.NULL_PARAMETER);
        }
        if (!FormatValidUtils.validPassword((String)password)) {
            throw new NulsRuntimeException(AccountErrorCode.PASSWORD_FORMAT_WRONG);
        }
        Account account = this.getAccountByAddress(chainId, address);
        if (account == null) {
            LoggerUtil.LOG.debug("the account isn't exist,chainId:{},address:{}", new Object[]{chainId, address});
            throw new NulsRuntimeException(AccountErrorCode.ACCOUNT_NOT_EXIST);
        }
        if (account.isEncrypted()) {
            LoggerUtil.LOG.debug("the account has encrypted,chainId:{},address:{}", new Object[]{chainId, address});
            throw new NulsRuntimeException(AccountErrorCode.ACCOUNT_IS_ALREADY_ENCRYPTED);
        }
        try {
            account.encrypt(password);
        }
        catch (Exception e) {
            LoggerUtil.LOG.error("encrypt the account occur exception,chainId:{},address:{}", new Object[]{chainId, address, e});
        }
        AccountPO po = new AccountPO(account);
        boolean result = this.accountStorageService.saveAccount(po);
        if (!result) {
            LoggerUtil.LOG.debug("save the account failed,chainId:{},address:{}", new Object[]{chainId, address});
        }
        this.keyStoreService.backupAccountToKeyStore(null, chainId, account.getAddress().getBase58(), password);
        return result;
    }

    @Override
    public boolean changePassword(int chainId, String address, String oldPassword, String newPassword) {
        if (!AddressTool.validAddress((int)chainId, (String)address)) {
            throw new NulsRuntimeException(AccountErrorCode.ADDRESS_ERROR);
        }
        if (StringUtils.isBlank((String)oldPassword) || StringUtils.isBlank((String)newPassword)) {
            throw new NulsRuntimeException(AccountErrorCode.PARAMETER_ERROR);
        }
        if (!FormatValidUtils.validPassword((String)oldPassword) || !FormatValidUtils.validPassword((String)newPassword)) {
            throw new NulsRuntimeException(AccountErrorCode.PASSWORD_FORMAT_WRONG);
        }
        Account account = this.getAccountByAddress(chainId, address);
        if (null == account) {
            throw new NulsRuntimeException(AccountErrorCode.ACCOUNT_NOT_EXIST);
        }
        if (!account.isEncrypted()) {
            throw new NulsRuntimeException(AccountErrorCode.ACCOUNT_UNENCRYPTED);
        }
        if (!account.validatePassword(oldPassword)) {
            throw new NulsRuntimeException(AccountErrorCode.PASSWORD_IS_WRONG);
        }
        try {
            account.unlock(oldPassword);
            account.encrypt(newPassword, true);
            AccountPO po = new AccountPO(account);
            boolean result = this.accountStorageService.updateAccount(po);
            this.accountCacheService.getLocalAccountMaps().put(account.getAddress().getBase58(), account);
            this.keyStoreService.backupAccountToKeyStore(null, chainId, account.getAddress().getBase58(), newPassword);
            HashMap<String, String> eventData = new HashMap<String, String>();
            eventData.put("address", account.getAddress().getBase58());
            EventCall.sendEvent("evt_ac_updatePassword", JSONUtils.obj2json(eventData));
            return result;
        }
        catch (NulsException e) {
            LoggerUtil.LOG.error(e);
            throw new NulsRuntimeException(e.getErrorCode());
        }
        catch (Exception e) {
            LoggerUtil.LOG.error(e);
            throw new NulsRuntimeException(AccountErrorCode.FAILED);
        }
    }

    @Override
    public String setOfflineAccountPassword(int chainId, String address, String priKey, String password) {
        if (StringUtils.isBlank((String)address) || !AddressTool.validAddress((int)chainId, (String)address)) {
            throw new NulsRuntimeException(AccountErrorCode.ADDRESS_ERROR);
        }
        if (StringUtils.isBlank((String)priKey) || !ECKey.isValidPrivteHex((String)priKey)) {
            throw new NulsRuntimeException(AccountErrorCode.PRIVATE_KEY_WRONG);
        }
        if (!FormatValidUtils.validPassword((String)password)) {
            throw new NulsRuntimeException(AccountErrorCode.PASSWORD_FORMAT_WRONG);
        }
        try {
            Account account = AccountTool.createAccount(chainId, priKey);
            if (!address.equals(account.getAddress().getBase58())) {
                throw new NulsRuntimeException(AccountErrorCode.ADDRESS_ERROR);
            }
            account.encrypt(password);
            return HexUtil.encode((byte[])account.getEncryptedPriKey());
        }
        catch (NulsException e) {
            throw new NulsRuntimeException(e.getErrorCode());
        }
    }

    @Override
    public String changeOfflinePassword(int chainId, String address, String priKey, String oldPassword, String newPassword) {
        if (StringUtils.isBlank((String)address) || !AddressTool.validAddress((int)chainId, (String)address)) {
            throw new NulsRuntimeException(AccountErrorCode.ADDRESS_ERROR);
        }
        if (StringUtils.isBlank((String)priKey)) {
            throw new NulsRuntimeException(AccountErrorCode.PARAMETER_ERROR);
        }
        if (!FormatValidUtils.validPassword((String)oldPassword)) {
            throw new NulsRuntimeException(AccountErrorCode.PASSWORD_FORMAT_WRONG);
        }
        if (!FormatValidUtils.validPassword((String)newPassword)) {
            throw new NulsRuntimeException(AccountErrorCode.PASSWORD_FORMAT_WRONG);
        }
        try {
            byte[] priKeyBytes = AESEncrypt.decrypt((byte[])HexUtil.decode((String)priKey), (String)oldPassword);
            if (!ECKey.isValidPrivteHex((String)HexUtil.encode((byte[])priKeyBytes))) {
                throw new NulsRuntimeException(AccountErrorCode.PRIVATE_KEY_WRONG);
            }
            Account account = AccountTool.createAccount(chainId, HexUtil.encode((byte[])priKeyBytes));
            if (!address.equals(account.getAddress().getBase58())) {
                throw new NulsRuntimeException(AccountErrorCode.ADDRESS_ERROR);
            }
            account.encrypt(newPassword);
            return HexUtil.encode((byte[])account.getEncryptedPriKey());
        }
        catch (NulsException e) {
            throw new NulsRuntimeException(e.getErrorCode());
        }
        catch (CryptoException e) {
            throw new NulsRuntimeException(AccountErrorCode.PASSWORD_IS_WRONG);
        }
    }

    @Override
    public boolean isEncrypted(int chainId, String address) {
        if (!AddressTool.validAddress((int)chainId, (String)address)) {
            LoggerUtil.LOG.debug("the address is illegal,chainId:{},address:{}", new Object[]{chainId, address});
            throw new NulsRuntimeException(AccountErrorCode.ADDRESS_ERROR);
        }
        Account account = this.getAccountByAddress(chainId, address);
        if (account == null) {
            LoggerUtil.LOG.debug("the account isn't exist,chainId:{},address:{}", new Object[]{chainId, address});
            throw new NulsRuntimeException(AccountErrorCode.ACCOUNT_NOT_EXIST);
        }
        boolean result = account.isEncrypted();
        LoggerUtil.LOG.debug("the account is Encrypted:{},chainId:{},address:{}", new Object[]{result, chainId, address});
        return result;
    }

    @Override
    public boolean removeAccount(int chainId, String address, String password) {
        boolean result;
        if (!AddressTool.validAddress((int)chainId, (String)address)) {
            throw new NulsRuntimeException(AccountErrorCode.ADDRESS_ERROR);
        }
        Account account = this.getAccountByAddress(chainId, address);
        if (account == null) {
            throw new NulsRuntimeException(AccountErrorCode.ACCOUNT_NOT_EXIST);
        }
        if (!account.validatePassword(password)) {
            throw new NulsRuntimeException(AccountErrorCode.PASSWORD_IS_WRONG);
        }
        try {
            result = this.accountStorageService.removeAccount(account.getAddress());
            this.accountCacheService.getLocalAccountMaps().remove(account.getAddress().getBase58());
            HashMap<String, String> eventData = new HashMap<String, String>();
            eventData.put("address", account.getAddress().getBase58());
            EventCall.sendEvent("evt_ac_removeAccount", JSONUtils.obj2json(eventData));
        }
        catch (Exception e) {
            LoggerUtil.LOG.error(e);
            throw new NulsRuntimeException(AccountErrorCode.FAILED);
        }
        return result;
    }

    @Override
    public boolean setRemark(int chainId, String address, String remark) {
        Account account = this.getAccountByAddress(chainId, address);
        if (null == account) {
            throw new NulsRuntimeException(AccountErrorCode.ACCOUNT_NOT_EXIST);
        }
        if (StringUtils.isBlank((String)remark)) {
            remark = null;
        }
        if (!FormatValidUtils.validRemark((String)remark)) {
            throw new NulsRuntimeException(AccountErrorCode.REMARK_TOO_LONG);
        }
        account.setRemark(remark);
        boolean result = this.accountStorageService.updateAccount(new AccountPO(account));
        this.accountCacheService.getLocalAccountMaps().put(account.getAddress().getBase58(), account);
        return result;
    }

    @Override
    public String getPrivateKey(int chainId, String address, String password) {
        Account account = this.getAccountByAddress(chainId, address);
        return this.getPrivateKey(chainId, account, password);
    }

    @Override
    public String getPrivateKey(int chainId, Account account, String password) {
        Preconditions.checkNotNull(account, AccountErrorCode.ACCOUNT_NOT_EXIST);
        if (account.isEncrypted()) {
            try {
                byte[] priKeyBytes = account.getPriKey(password);
                return HexUtil.encode((byte[])priKeyBytes);
            }
            catch (NulsException e) {
                throw new NulsRuntimeException(AccountErrorCode.PASSWORD_IS_WRONG, "chainId=" + chainId + ", account=" + account.getAddress().getBase58() + ", pwd=" + password + ".");
            }
        }
        return null;
    }

    @Override
    public String getPublicKey(int chainId, String address, String password) {
        Account account = this.getAccountByAddress(chainId, address);
        if (null == account) {
            throw new NulsRuntimeException(AccountErrorCode.ACCOUNT_NOT_EXIST);
        }
        if (account.isEncrypted()) {
            byte[] pubKeyBytes = account.getPubKey();
            return HexUtil.encode((byte[])pubKeyBytes);
        }
        return null;
    }

    @Override
    public List<String> getAllPrivateKey(int chainId, String password) {
        List<Account> localAccountList = this.getAccountList();
        if (localAccountList == null || localAccountList.isEmpty()) {
            throw new NulsRuntimeException(AccountErrorCode.ACCOUNT_NOT_EXIST);
        }
        if (chainId > 0) {
            ArrayList chainAccountList = new ArrayList();
            localAccountList.stream().filter(p -> p.getChainId() == chainId).forEach(account -> chainAccountList.add(account));
            localAccountList.clear();
            localAccountList.addAll(chainAccountList);
        }
        if (!FormatValidUtils.validPassword((String)password)) {
            throw new NulsRuntimeException(AccountErrorCode.PASSWORD_IS_WRONG);
        }
        ArrayList<String> list = new ArrayList<String>();
        for (Account account2 : localAccountList) {
            if (account2.isEncrypted()) {
                if (StringUtils.isBlank((String)password)) {
                    throw new NulsRuntimeException(AccountErrorCode.HAVE_ENCRYPTED_ACCOUNT);
                }
                try {
                    byte[] priKeyBytes = account2.getPriKey(password);
                    list.add(HexUtil.encode((byte[])priKeyBytes));
                    continue;
                }
                catch (NulsException e) {
                    throw new NulsRuntimeException(AccountErrorCode.PASSWORD_IS_WRONG);
                }
            }
            if (StringUtils.isNotBlank((String)password)) {
                throw new NulsRuntimeException(AccountErrorCode.HAVE_UNENCRYPTED_ACCOUNT);
            }
            list.add(HexUtil.encode((byte[])account2.getPriKey()));
        }
        return list;
    }

    @Override
    public Account importAccountByPrikey(Chain chain, String prikey, String password, boolean overwrite) throws NulsException {
        Account account;
        Address address;
        Account account2;
        int chainId = chain.getChainId();
        if (!ECKey.isValidPrivteHex((String)prikey)) {
            throw new NulsRuntimeException(AccountErrorCode.PRIVATE_KEY_WRONG);
        }
        if (!FormatValidUtils.validPassword((String)password)) {
            throw new NulsRuntimeException(AccountErrorCode.PASSWORD_IS_WRONG);
        }
        if (!overwrite && null != (account2 = this.getAccountByAddress(chainId, (address = AccountTool.newAddress(chainId, prikey)).getBase58()))) {
            throw new NulsRuntimeException(AccountErrorCode.ACCOUNT_EXIST);
        }
        try {
            account = AccountTool.createAccount(chainId, prikey);
        }
        catch (NulsException e) {
            throw new NulsRuntimeException(AccountErrorCode.PRIVATE_KEY_WRONG);
        }
        account.encrypt(password);
        Account acc = this.getAccountByAddress(chainId, account.getAddress().getBase58());
        if (null == acc) {
            account.setAlias(this.aliasService.getAliasByAddress(chainId, account.getAddress().getBase58()));
        } else {
            account.setAlias(acc.getAlias());
        }
        this.accountStorageService.saveAccount(new AccountPO(account));
        this.accountCacheService.getLocalAccountMaps().put(account.getAddress().getBase58(), account);
        this.keyStoreService.backupAccountToKeyStore(null, chainId, account.getAddress().getBase58(), password);
        return account;
    }

    @Override
    public Account importAccountByKeyStore(AccountKeyStore keyStore, Chain chain, String password, boolean overwrite) throws NulsException {
        Account acc;
        byte[] newAccountHash160;
        Account account;
        int chainId = chain.getChainId();
        if (null == keyStore || StringUtils.isBlank((String)keyStore.getAddress())) {
            throw new NulsRuntimeException(AccountErrorCode.PARAMETER_ERROR);
        }
        if (!FormatValidUtils.validPassword((String)password)) {
            throw new NulsRuntimeException(AccountErrorCode.PASSWORD_IS_WRONG);
        }
        if (!overwrite && null != (account = this.getAccountByAddress(chainId, keyStore.getAddress()))) {
            throw new NulsRuntimeException(AccountErrorCode.ACCOUNT_EXIST);
        }
        if (null != keyStore.getPrikey() && keyStore.getPrikey().length > 0) {
            if (!ECKey.isValidPrivteHex((String)HexUtil.encode((byte[])keyStore.getPrikey()))) {
                throw new NulsRuntimeException(AccountErrorCode.PARAMETER_ERROR);
            }
            byte[] priKey = keyStore.getPrikey();
            account = AccountTool.createAccount(chainId, HexUtil.encode((byte[])priKey));
            if (!account.getAddress().getBase58().equals(keyStore.getAddress())) {
                try {
                    newAccountHash160 = account.getAddress().getHash160();
                    byte[] bytes = Base58.decode((String)keyStore.getAddress());
                    byte[] originalAddressHash160 = new byte[20];
                    System.arraycopy(bytes, 3, originalAddressHash160, 0, 20);
                    if (!Arrays.equals(newAccountHash160, originalAddressHash160)) {
                        throw new NulsRuntimeException(AccountErrorCode.ACCOUNTKEYSTORE_FILE_DAMAGED);
                    }
                }
                catch (Exception e) {
                    throw new NulsRuntimeException(AccountErrorCode.ACCOUNTKEYSTORE_FILE_DAMAGED);
                }
            }
        } else if (null == keyStore.getPrikey() && null != keyStore.getEncryptedPrivateKey()) {
            try {
                byte[] priKey = AESEncrypt.decrypt((byte[])HexUtil.decode((String)keyStore.getEncryptedPrivateKey()), (String)password);
                account = AccountTool.createAccount(chainId, HexUtil.encode((byte[])priKey));
            }
            catch (CryptoException e) {
                throw new NulsRuntimeException(AccountErrorCode.PASSWORD_IS_WRONG);
            }
            if (!account.getAddress().getBase58().equals(keyStore.getAddress())) {
                try {
                    newAccountHash160 = account.getAddress().getHash160();
                    byte[] bytes = Base58.decode((String)keyStore.getAddress());
                    byte[] originalAddressHash160 = new byte[20];
                    System.arraycopy(bytes, 3, originalAddressHash160, 0, 20);
                    if (!Arrays.equals(newAccountHash160, originalAddressHash160)) {
                        throw new NulsRuntimeException(AccountErrorCode.ACCOUNTKEYSTORE_FILE_DAMAGED);
                    }
                }
                catch (Exception e) {
                    throw new NulsRuntimeException(AccountErrorCode.ACCOUNTKEYSTORE_FILE_DAMAGED);
                }
            }
        } else {
            throw new NulsRuntimeException(AccountErrorCode.PARAMETER_ERROR);
        }
        if (null == (acc = this.getAccountByAddress(chainId, account.getAddress().getBase58()))) {
            account.setAlias(this.aliasService.getAliasByAddress(chainId, account.getAddress().getBase58()));
        } else {
            account.setAlias(acc.getAlias());
        }
        account.encrypt(password);
        this.accountStorageService.saveAccount(new AccountPO(account));
        this.accountCacheService.getLocalAccountMaps().put(account.getAddress().getBase58(), account);
        this.keyStoreService.backupAccountToKeyStore(null, chainId, account.getAddress().getBase58(), password);
        return account;
    }

    @Override
    public void importAccountListByKeystore(List<AccountKeyStore> keyStoreList, Chain chain) throws NulsException {
        int chainId = chain.getChainId();
        try {
            for (AccountKeyStore keyStore : keyStoreList) {
                Account account = AccountTool.createAccountByPubKey(chainId, keyStore.getEncryptedPrivateKey(), keyStore.getPubKey());
                account.setAlias(this.aliasService.getAliasByAddress(chainId, account.getAddress().getBase58()));
                this.accountStorageService.saveAccount(new AccountPO(account));
                this.accountCacheService.getLocalAccountMaps().put(account.getAddress().getBase58(), account);
            }
        }
        catch (Exception e) {
            LoggerUtil.LOG.error(e.getMessage());
            throw new NulsException(AccountErrorCode.SYS_UNKOWN_EXCEPTION);
        }
    }

    @Override
    public P2PHKSignature signDigest(byte[] digest, int chainId, String address, String password) throws NulsException {
        if (null == digest || digest.length == 0) {
            throw new NulsRuntimeException(AccountErrorCode.PARAMETER_ERROR);
        }
        Account account = this.getAccountByAddress(chainId, address);
        if (null == account) {
            throw new NulsRuntimeException(AccountErrorCode.ACCOUNT_NOT_EXIST);
        }
        ECKey ecKey = account.getEcKey(password);
        try {
            byte[] signBytes = SignatureUtil.signDigest((byte[])digest, (ECKey)ecKey).serialize();
            return new P2PHKSignature(signBytes, ecKey.getPubKey());
        }
        catch (IOException e) {
            LoggerUtil.LOG.error(e.getMessage());
            throw new NulsRuntimeException(AccountErrorCode.IO_ERROR);
        }
    }

    @Override
    public BlockSignature signBlockDigest(byte[] digest, int chainId, String address, String password) throws NulsException {
        if (null == digest || digest.length == 0) {
            throw new NulsRuntimeException(AccountErrorCode.PARAMETER_ERROR);
        }
        Account account = this.getAccountByAddress(chainId, address);
        if (null == account) {
            throw new NulsRuntimeException(AccountErrorCode.ACCOUNT_NOT_EXIST);
        }
        ECKey ecKey = account.getEcKey(password);
        NulsSignData signData = SignatureUtil.signDigest((byte[])digest, (ECKey)ecKey);
        BlockSignature blockSign = new BlockSignature();
        blockSign.setSignData(signData);
        blockSign.setPublicKey(ecKey.getPubKey());
        return blockSign;
    }
}

