/*
 * Decompiled with CFR 0.152.
 */
package org.ethereum.db;

import java.io.Serializable;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.bouncycastle.util.Arrays;
import org.ethereum.core.Block;
import org.ethereum.core.BlockHeader;
import org.ethereum.crypto.HashUtil;
import org.ethereum.datasource.DataSourceArray;
import org.ethereum.datasource.ObjectDataSource;
import org.ethereum.datasource.Serializer;
import org.ethereum.datasource.Source;
import org.ethereum.db.AbstractBlockstore;
import org.ethereum.util.ByteUtil;
import org.ethereum.util.FastByteComparisons;
import org.ethereum.util.RLP;
import org.ethereum.util.RLPElement;
import org.ethereum.util.RLPList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexedBlockStore
extends AbstractBlockstore {
    private static final Logger logger = LoggerFactory.getLogger((String)"general");
    Source<byte[], byte[]> indexDS;
    DataSourceArray<List<BlockInfo>> index;
    Source<byte[], byte[]> blocksDS;
    ObjectDataSource<Block> blocks;
    public static final Serializer<List<BlockInfo>, byte[]> BLOCK_INFO_SERIALIZER = new Serializer<List<BlockInfo>, byte[]>(){

        @Override
        public byte[] serialize(List<BlockInfo> value) {
            ArrayList<byte[]> rlpBlockInfoList = new ArrayList<byte[]>();
            for (BlockInfo blockInfo : value) {
                byte[] hash = RLP.encodeElement(blockInfo.getHash());
                if (blockInfo.getTotalDifficulty() == null || blockInfo.getTotalDifficulty().compareTo(BigInteger.ZERO) < 0) {
                    throw new RuntimeException("BlockInfo totalDifficulty should be positive BigInteger");
                }
                byte[] totalDiff = RLP.encodeBigInteger(blockInfo.getTotalDifficulty());
                byte[] isMainChain = RLP.encodeInt(blockInfo.isMainChain() ? 1 : 0);
                rlpBlockInfoList.add(RLP.encodeList(hash, totalDiff, isMainChain));
            }
            byte[][] elements = (byte[][])rlpBlockInfoList.toArray((T[])new byte[rlpBlockInfoList.size()][]);
            return RLP.encodeList(elements);
        }

        @Override
        public List<BlockInfo> deserialize(byte[] bytes) {
            if (bytes == null) {
                return null;
            }
            ArrayList<BlockInfo> blockInfoList = new ArrayList<BlockInfo>();
            RLPList list = (RLPList)RLP.decode2(bytes).get(0);
            for (RLPElement element : list) {
                RLPList rlpBlock = (RLPList)element;
                BlockInfo blockInfo = new BlockInfo();
                byte[] rlpHash = ((RLPElement)rlpBlock.get(0)).getRLPData();
                blockInfo.setHash(rlpHash == null ? new byte[]{} : rlpHash);
                byte[] rlpTotalDiff = ((RLPElement)rlpBlock.get(1)).getRLPData();
                blockInfo.setTotalDifficulty(rlpTotalDiff == null ? BigInteger.ZERO : ByteUtil.bytesToBigInteger(rlpTotalDiff));
                blockInfo.setMainChain(ByteUtil.byteArrayToInt(((RLPElement)rlpBlock.get(2)).getRLPData()) == 1);
                blockInfoList.add(blockInfo);
            }
            return blockInfoList;
        }
    };

    public void init(Source<byte[], byte[]> index, Source<byte[], byte[]> blocks) {
        this.indexDS = index;
        this.index = new DataSourceArray<List<BlockInfo>>(new ObjectDataSource<List<BlockInfo>>(index, BLOCK_INFO_SERIALIZER, 512));
        this.blocksDS = blocks;
        this.blocks = new ObjectDataSource<Block>(blocks, new Serializer<Block, byte[]>(){

            @Override
            public byte[] serialize(Block block) {
                return block.getEncoded();
            }

            @Override
            public Block deserialize(byte[] bytes) {
                return bytes == null ? null : new Block(bytes);
            }
        }, 256);
    }

    @Override
    public synchronized Block getBestBlock() {
        Long maxLevel = this.getMaxNumber();
        if (maxLevel < 0L) {
            return null;
        }
        Block bestBlock = this.getChainBlockByNumber(maxLevel);
        if (bestBlock != null) {
            return bestBlock;
        }
        while (bestBlock == null) {
            maxLevel = maxLevel - 1L;
            bestBlock = this.getChainBlockByNumber(maxLevel);
        }
        return bestBlock;
    }

    @Override
    public synchronized byte[] getBlockHashByNumber(long blockNumber) {
        Block chainBlock = this.getChainBlockByNumber(blockNumber);
        return chainBlock == null ? null : chainBlock.getHash();
    }

    @Override
    public synchronized void flush() {
        this.blocks.flush();
        this.index.flush();
        this.blocksDS.flush();
        this.indexDS.flush();
    }

    @Override
    public synchronized void saveBlock(Block block, BigInteger totalDifficulty, boolean mainChain) {
        this.addInternalBlock(block, totalDifficulty, mainChain);
    }

    private void addInternalBlock(Block block, BigInteger totalDifficulty, boolean mainChain) {
        ArrayList<BlockInfo> blockInfos = block.getNumber() >= (long)this.index.size() ? null : this.index.get((int)block.getNumber());
        blockInfos = blockInfos == null ? new ArrayList<BlockInfo>() : blockInfos;
        BlockInfo blockInfo = new BlockInfo();
        blockInfo.setTotalDifficulty(totalDifficulty);
        blockInfo.setHash(block.getHash());
        blockInfo.setMainChain(mainChain);
        this.putBlockInfo(blockInfos, blockInfo);
        this.index.set((int)block.getNumber(), (List<BlockInfo>)blockInfos);
        this.blocks.put(block.getHash(), block);
    }

    private void putBlockInfo(List<BlockInfo> blockInfos, BlockInfo blockInfo) {
        for (int i = 0; i < blockInfos.size(); ++i) {
            BlockInfo curBlockInfo = blockInfos.get(i);
            if (!FastByteComparisons.equal(curBlockInfo.getHash(), blockInfo.getHash())) continue;
            blockInfos.set(i, blockInfo);
            return;
        }
        blockInfos.add(blockInfo);
    }

    public synchronized List<Block> getBlocksByNumber(long number) {
        ArrayList<Block> result = new ArrayList<Block>();
        if (number >= (long)this.index.size()) {
            return result;
        }
        List<BlockInfo> blockInfos = this.index.get((int)number);
        if (blockInfos == null) {
            return result;
        }
        for (BlockInfo blockInfo : blockInfos) {
            byte[] hash = blockInfo.getHash();
            Block block = (Block)this.blocks.get(hash);
            result.add(block);
        }
        return result;
    }

    @Override
    public synchronized Block getChainBlockByNumber(long number) {
        if (number >= (long)this.index.size()) {
            return null;
        }
        List<BlockInfo> blockInfos = this.index.get((int)number);
        if (blockInfos == null) {
            return null;
        }
        for (BlockInfo blockInfo : blockInfos) {
            if (!blockInfo.isMainChain()) continue;
            byte[] hash = blockInfo.getHash();
            return (Block)this.blocks.get(hash);
        }
        return null;
    }

    @Override
    public synchronized Block getBlockByHash(byte[] hash) {
        return (Block)this.blocks.get(hash);
    }

    @Override
    public synchronized boolean isBlockExist(byte[] hash) {
        return this.blocks.get(hash) != null;
    }

    @Override
    public synchronized BigInteger getTotalDifficultyForHash(byte[] hash) {
        Block block = this.getBlockByHash(hash);
        if (block == null) {
            return BigInteger.ZERO;
        }
        Long level = block.getNumber();
        List<BlockInfo> blockInfos = this.index.get(level.intValue());
        for (BlockInfo blockInfo : blockInfos) {
            if (!Arrays.areEqual((byte[])blockInfo.getHash(), (byte[])hash)) continue;
            return blockInfo.totalDifficulty;
        }
        return BigInteger.ZERO;
    }

    @Override
    public synchronized BigInteger getTotalDifficulty() {
        BlockInfo blockInfo;
        long maxNumber = this.getMaxNumber();
        List<BlockInfo> blockInfos = this.index.get((int)maxNumber);
        for (BlockInfo blockInfo2 : blockInfos) {
            if (!blockInfo2.isMainChain()) continue;
            return blockInfo2.getTotalDifficulty();
        }
        block1: while (true) {
            List<BlockInfo> infos = this.getBlockInfoForLevel(--maxNumber);
            Iterator<BlockInfo> iterator = infos.iterator();
            do {
                if (!iterator.hasNext()) continue block1;
            } while (!(blockInfo = iterator.next()).isMainChain());
            break;
        }
        return blockInfo.getTotalDifficulty();
    }

    @Override
    public synchronized long getMaxNumber() {
        Long bestIndex = 0L;
        if (this.index.size() > 0) {
            bestIndex = this.index.size();
        }
        return bestIndex - 1L;
    }

    @Override
    public synchronized List<byte[]> getListHashesEndWith(byte[] hash, long number) {
        List<Block> blocks = this.getListBlocksEndWith(hash, number);
        ArrayList<byte[]> hashes = new ArrayList<byte[]>(blocks.size());
        for (Block b : blocks) {
            hashes.add(b.getHash());
        }
        return hashes;
    }

    @Override
    public synchronized List<BlockHeader> getListHeadersEndWith(byte[] hash, long qty) {
        List<Block> blocks = this.getListBlocksEndWith(hash, qty);
        ArrayList<BlockHeader> headers = new ArrayList<BlockHeader>(blocks.size());
        for (Block b : blocks) {
            headers.add(b.getHeader());
        }
        return headers;
    }

    @Override
    public synchronized List<Block> getListBlocksEndWith(byte[] hash, long qty) {
        return this.getListBlocksEndWithInner(hash, qty);
    }

    private List<Block> getListBlocksEndWithInner(byte[] hash, long qty) {
        Block block = (Block)this.blocks.get(hash);
        if (block == null) {
            return new ArrayList<Block>();
        }
        ArrayList<Block> blocks = new ArrayList<Block>((int)qty);
        int i = 0;
        while ((long)i < qty) {
            blocks.add(block);
            block = (Block)this.blocks.get(block.getParentHash());
            if (block == null) break;
            ++i;
        }
        return blocks;
    }

    @Override
    public synchronized void reBranch(Block forkBlock) {
        long currentLevel;
        Block bestBlock = this.getBestBlock();
        Block forkLine = forkBlock;
        if (forkBlock.getNumber() > bestBlock.getNumber()) {
            long maxLevel;
            for (currentLevel = maxLevel = Math.max(bestBlock.getNumber(), forkBlock.getNumber()); currentLevel > bestBlock.getNumber(); --currentLevel) {
                List<BlockInfo> blocks = this.getBlockInfoForLevel(currentLevel);
                BlockInfo blockInfo = IndexedBlockStore.getBlockInfoForHash(blocks, forkLine.getHash());
                if (blockInfo != null) {
                    blockInfo.setMainChain(true);
                    this.setBlockInfoForLevel(currentLevel, blocks);
                }
                forkLine = this.getBlockByHash(forkLine.getParentHash());
            }
        }
        Block bestLine = bestBlock;
        if (bestBlock.getNumber() > forkBlock.getNumber()) {
            while (currentLevel > forkBlock.getNumber()) {
                List<BlockInfo> blocks = this.getBlockInfoForLevel(currentLevel);
                BlockInfo blockInfo = IndexedBlockStore.getBlockInfoForHash(blocks, bestLine.getHash());
                if (blockInfo != null) {
                    blockInfo.setMainChain(false);
                    this.setBlockInfoForLevel(currentLevel, blocks);
                }
                bestLine = this.getBlockByHash(bestLine.getParentHash());
                --currentLevel;
            }
        }
        while (!bestLine.isEqual(forkLine)) {
            BlockInfo forkInfo;
            List<BlockInfo> levelBlocks = this.getBlockInfoForLevel(currentLevel);
            BlockInfo bestInfo = IndexedBlockStore.getBlockInfoForHash(levelBlocks, bestLine.getHash());
            if (bestInfo != null) {
                bestInfo.setMainChain(false);
                this.setBlockInfoForLevel(currentLevel, levelBlocks);
            }
            if ((forkInfo = IndexedBlockStore.getBlockInfoForHash(levelBlocks, forkLine.getHash())) != null) {
                forkInfo.setMainChain(true);
                this.setBlockInfoForLevel(currentLevel, levelBlocks);
            }
            bestLine = this.getBlockByHash(bestLine.getParentHash());
            forkLine = this.getBlockByHash(forkLine.getParentHash());
            --currentLevel;
        }
    }

    public synchronized List<byte[]> getListHashesStartWith(long number, long maxBlocks) {
        List<BlockInfo> blockInfos;
        ArrayList<byte[]> result = new ArrayList<byte[]>();
        int i = 0;
        while ((long)i < maxBlocks && (blockInfos = this.index.get((int)number)) != null) {
            for (BlockInfo blockInfo : blockInfos) {
                if (!blockInfo.isMainChain()) continue;
                result.add(blockInfo.getHash());
                break;
            }
            ++number;
            ++i;
        }
        maxBlocks -= (long)i;
        return result;
    }

    public synchronized void printChain() {
        Long number = this.getMaxNumber();
        int i = 0;
        while ((long)i < number) {
            List<BlockInfo> levelInfos = this.index.get(i);
            if (levelInfos != null) {
                System.out.print(i);
                for (BlockInfo blockInfo : levelInfos) {
                    if (blockInfo.isMainChain()) {
                        System.out.print(" [" + HashUtil.shortHash(blockInfo.getHash()) + "] ");
                        continue;
                    }
                    System.out.print(" " + HashUtil.shortHash(blockInfo.getHash()) + " ");
                }
                System.out.println();
            }
            ++i;
        }
    }

    private synchronized List<BlockInfo> getBlockInfoForLevel(long level) {
        return this.index.get((int)level);
    }

    private synchronized void setBlockInfoForLevel(long level, List<BlockInfo> infos) {
        this.index.set((int)level, infos);
    }

    private static BlockInfo getBlockInfoForHash(List<BlockInfo> blocks, byte[] hash) {
        for (BlockInfo blockInfo : blocks) {
            if (!Arrays.areEqual((byte[])hash, (byte[])blockInfo.getHash())) continue;
            return blockInfo;
        }
        return null;
    }

    @Override
    public synchronized void load() {
    }

    @Override
    public synchronized void close() {
    }

    public static class BlockInfo
    implements Serializable {
        byte[] hash;
        BigInteger totalDifficulty;
        boolean mainChain;

        public byte[] getHash() {
            return this.hash;
        }

        public void setHash(byte[] hash) {
            this.hash = hash;
        }

        public BigInteger getTotalDifficulty() {
            return this.totalDifficulty;
        }

        public void setTotalDifficulty(BigInteger totalDifficulty) {
            this.totalDifficulty = totalDifficulty;
        }

        public boolean isMainChain() {
            return this.mainChain;
        }

        public void setMainChain(boolean mainChain) {
            this.mainChain = mainChain;
        }
    }
}

