/*
 * Decompiled with CFR 0.152.
 */
package io.nuls.base.script;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import io.nuls.base.basic.AddressTool;
import io.nuls.base.data.Transaction;
import io.nuls.base.script.ScriptBuilder;
import io.nuls.base.script.ScriptChunk;
import io.nuls.base.script.ScriptException;
import io.nuls.core.crypto.ECKey;
import io.nuls.core.crypto.HexUtil;
import io.nuls.core.crypto.Sha256Hash;
import io.nuls.core.crypto.UnsafeByteArrayOutputStream;
import io.nuls.core.model.ByteUtils;
import io.nuls.core.model.CollectionUtils;
import io.nuls.core.parse.SerializeUtils;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.math.BigInteger;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import javax.annotation.Nullable;
import org.bouncycastle.crypto.digests.RIPEMD160Digest;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Script {
    public static final EnumSet<VerifyFlag> ALL_VERIFY_FLAGS = EnumSet.allOf(VerifyFlag.class);
    private static final Logger log = LoggerFactory.getLogger(Script.class);
    public static final long MAX_SCRIPT_ELEMENT_SIZE = 520L;
    public static final int SIG_SIZE = 75;
    public static final int MAX_P2SH_SIGOPS = 15;
    protected List<ScriptChunk> chunks;
    protected byte[] program;
    private long creationTimeSeconds;
    private static final ScriptChunk[] STANDARD_TRANSACTION_SCRIPT_CHUNKS = new ScriptChunk[]{new ScriptChunk(118, null, 0), new ScriptChunk(169, null, 1), new ScriptChunk(136, null, 23), new ScriptChunk(172, null, 24)};

    private Script() {
        this.chunks = Lists.newArrayList();
    }

    Script(List<ScriptChunk> chunks) {
        this.chunks = Collections.unmodifiableList(new ArrayList<ScriptChunk>(chunks));
        this.creationTimeSeconds = System.currentTimeMillis() / 1000L;
    }

    public Script(byte[] programBytes) throws ScriptException {
        this.program = programBytes;
        this.parse(programBytes);
        this.creationTimeSeconds = 0L;
    }

    public Script(byte[] programBytes, long creationTimeSeconds) throws ScriptException {
        this.program = programBytes;
        this.parse(programBytes);
        this.creationTimeSeconds = creationTimeSeconds;
    }

    public long getCreationTimeSeconds() {
        return this.creationTimeSeconds;
    }

    public void setCreationTimeSeconds(long creationTimeSeconds) {
        this.creationTimeSeconds = creationTimeSeconds;
    }

    public String toString() {
        return CollectionUtils.join(this.chunks);
    }

    public byte[] getProgram() {
        try {
            if (this.program != null) {
                return Arrays.copyOf(this.program, this.program.length);
            }
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            for (ScriptChunk chunk : this.chunks) {
                chunk.write(bos);
            }
            this.program = bos.toByteArray();
            return this.program;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public List<ScriptChunk> getChunks() {
        return Collections.unmodifiableList(this.chunks);
    }

    private void parse(byte[] program) throws ScriptException {
        this.chunks = new ArrayList<ScriptChunk>(5);
        ByteArrayInputStream bis = new ByteArrayInputStream(program);
        int initialSize = bis.available();
        while (bis.available() > 0) {
            ScriptChunk chunk;
            int startLocationInProgram = initialSize - bis.available();
            int opcode = bis.read();
            long dataToRead = -1L;
            if (opcode >= 0 && opcode < 76) {
                dataToRead = opcode;
            } else if (opcode == 76) {
                if (bis.available() < 1) {
                    throw new ScriptException("Unexpected end of script");
                }
                dataToRead = bis.read();
            } else if (opcode == 77) {
                if (bis.available() < 2) {
                    throw new ScriptException("Unexpected end of script");
                }
                dataToRead = bis.read() | bis.read() << 8;
            } else if (opcode == 78) {
                if (bis.available() < 4) {
                    throw new ScriptException("Unexpected end of script");
                }
                dataToRead = (long)bis.read() | (long)bis.read() << 8 | (long)bis.read() << 16 | (long)bis.read() << 24;
            }
            if (dataToRead == -1L) {
                chunk = new ScriptChunk(opcode, null, startLocationInProgram);
            } else {
                if (dataToRead > (long)bis.available()) {
                    throw new ScriptException("Push of entity element that is larger than remaining entity");
                }
                byte[] data = new byte[(int)dataToRead];
                Preconditions.checkState((dataToRead == 0L || (long)bis.read(data, 0, (int)dataToRead) == dataToRead ? 1 : 0) != 0);
                chunk = new ScriptChunk(opcode, data, startLocationInProgram);
            }
            for (ScriptChunk c : STANDARD_TRANSACTION_SCRIPT_CHUNKS) {
                if (!c.equals(chunk)) continue;
                chunk = c;
            }
            this.chunks.add(chunk);
        }
    }

    public boolean isSentToRawPubKey() {
        return this.chunks.size() == 2 && this.chunks.get(1).equalsOpCode(172) && !this.chunks.get(0).isOpCode() && this.chunks.get((int)0).data.length > 1;
    }

    public boolean isSentToAddress() {
        return this.chunks.size() == 5 && this.chunks.get(0).equalsOpCode(118) && this.chunks.get(1).equalsOpCode(169) && this.chunks.get((int)2).data.length == 23 && this.chunks.get(3).equalsOpCode(136) && this.chunks.get(4).equalsOpCode(172);
    }

    @Deprecated
    public boolean isSentToP2SH() {
        return this.isPayToScriptHash();
    }

    public byte[] getPubKeyHash() throws ScriptException {
        if (this.isSentToAddress()) {
            return this.chunks.get((int)2).data;
        }
        if (this.isPayToScriptHash()) {
            return this.chunks.get((int)1).data;
        }
        throw new ScriptException("Script not in the standard scriptPubKey form");
    }

    public byte[] getPubKey() throws ScriptException {
        if (this.chunks.size() != 2) {
            throw new ScriptException("Script not of right size, expecting 2 but got " + this.chunks.size());
        }
        ScriptChunk chunk0 = this.chunks.get(0);
        byte[] chunk0data = chunk0.data;
        ScriptChunk chunk1 = this.chunks.get(1);
        byte[] chunk1data = chunk1.data;
        if (chunk0data != null && chunk0data.length > 2 && chunk1data != null && chunk1data.length > 2) {
            return chunk1data;
        }
        if (chunk1.equalsOpCode(172) && chunk0data != null && chunk0data.length > 2) {
            return chunk0data;
        }
        throw new ScriptException("Script did not match expected form: " + this);
    }

    public byte[] getCLTVPaymentChannelSenderPubKey() throws ScriptException {
        if (!this.isSentToCLTVPaymentChannel()) {
            throw new ScriptException("Script not a standard CHECKLOCKTIMVERIFY transaction: " + this);
        }
        return this.chunks.get((int)8).data;
    }

    public byte[] getCLTVPaymentChannelRecipientPubKey() throws ScriptException {
        if (!this.isSentToCLTVPaymentChannel()) {
            throw new ScriptException("Script not a standard CHECKLOCKTIMVERIFY transaction: " + this);
        }
        return this.chunks.get((int)1).data;
    }

    public BigInteger getCLTVPaymentChannelExpiry() {
        if (!this.isSentToCLTVPaymentChannel()) {
            throw new ScriptException("Script not a standard CHECKLOCKTIMEVERIFY transaction: " + this);
        }
        return Script.castToBigInteger(this.chunks.get((int)4).data, 5);
    }

    public static void writeBytes(OutputStream os, byte[] buf) throws IOException {
        if (buf.length < 76) {
            os.write(buf.length);
            os.write(buf);
        } else if (buf.length < 256) {
            os.write(76);
            os.write(buf.length);
            os.write(buf);
        } else if (buf.length < 65536) {
            os.write(77);
            os.write(0xFF & buf.length);
            os.write(0xFF & buf.length >> 8);
            os.write(buf);
        } else {
            throw new RuntimeException("Unimplemented");
        }
    }

    public static byte[] createMultiSigOutputScript(int threshold, List<ECKey> pubkeys) {
        Preconditions.checkArgument((threshold > 0 ? 1 : 0) != 0);
        Preconditions.checkArgument((threshold <= pubkeys.size() ? 1 : 0) != 0);
        Preconditions.checkArgument((pubkeys.size() <= 16 ? 1 : 0) != 0);
        if (pubkeys.size() > 3) {
            log.warn("Creating a multi-signature output that is non-standard: {} pubkeys, should be <= 3", (Object)pubkeys.size());
        }
        try {
            ByteArrayOutputStream bits = new ByteArrayOutputStream();
            bits.write(Script.encodeToOpN(threshold));
            for (ECKey key : pubkeys) {
                Script.writeBytes(bits, key.getPubKey());
            }
            bits.write(Script.encodeToOpN(pubkeys.size()));
            bits.write(174);
            return bits.toByteArray();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static byte[] createInputScript(byte[] signature, byte[] pubkey) {
        try {
            UnsafeByteArrayOutputStream bits = new UnsafeByteArrayOutputStream(signature.length + pubkey.length + 2);
            Script.writeBytes((OutputStream)bits, signature);
            Script.writeBytes((OutputStream)bits, pubkey);
            return bits.toByteArray();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static byte[] createInputScript(byte[] signature) {
        try {
            UnsafeByteArrayOutputStream bits = new UnsafeByteArrayOutputStream(signature.length + 2);
            Script.writeBytes((OutputStream)bits, signature);
            return bits.toByteArray();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public Script createEmptyInputScript(@Nullable ECKey key, @Nullable Script redeemScript) {
        if (this.isSentToAddress()) {
            Preconditions.checkArgument((key != null ? 1 : 0) != 0, (Object)"Key required to create pay-to-address input script");
            return ScriptBuilder.createInputScript(null, key);
        }
        if (this.isSentToRawPubKey()) {
            return ScriptBuilder.createInputScript(null);
        }
        if (this.isPayToScriptHash()) {
            Preconditions.checkArgument((redeemScript != null ? 1 : 0) != 0, (Object)"Redeem script required to create P2SH input script");
            return ScriptBuilder.createP2SHMultiSigInputScript(null, redeemScript);
        }
        throw new ScriptException("Do not understand script type: " + this);
    }

    public Script getScriptSigWithSignature(Script scriptSig, byte[] sigBytes, int index) {
        int sigsPrefixCount = 0;
        int sigsSuffixCount = 0;
        if (this.isPayToScriptHash()) {
            sigsPrefixCount = 1;
            sigsSuffixCount = 1;
        } else if (this.isSentToMultiSig()) {
            sigsPrefixCount = 1;
        } else if (this.isSentToAddress()) {
            sigsSuffixCount = 1;
        }
        return ScriptBuilder.updateScriptWithSignature(scriptSig, sigBytes, index, sigsPrefixCount, sigsSuffixCount);
    }

    public int getSigInsertionIndex(Sha256Hash hash, ECKey signingKey) {
        List<ScriptChunk> existingChunks = this.chunks.subList(1, this.chunks.size() - 1);
        ScriptChunk redeemScriptChunk = this.chunks.get(this.chunks.size() - 1);
        Preconditions.checkNotNull((Object)redeemScriptChunk.data);
        Script redeemScript = new Script(redeemScriptChunk.data);
        int sigCount = 0;
        int myIndex = redeemScript.findKeyInRedeem(signingKey);
        for (ScriptChunk chunk : existingChunks) {
            if (chunk.opcode == 0) continue;
            Preconditions.checkNotNull((Object)chunk.data);
            if (myIndex < redeemScript.findSigInRedeem(chunk.data, hash)) {
                return sigCount;
            }
            ++sigCount;
        }
        return sigCount;
    }

    private int findKeyInRedeem(ECKey key) {
        Preconditions.checkArgument((boolean)this.chunks.get(0).isOpCode());
        int numKeys = Script.decodeFromOpN(this.chunks.get((int)(this.chunks.size() - 2)).opcode);
        for (int i = 0; i < numKeys; ++i) {
            if (!Arrays.equals(this.chunks.get((int)(1 + i)).data, key.getPubKey())) continue;
            return i;
        }
        throw new IllegalStateException("Could not find matching key " + key.toString() + " in script " + this);
    }

    public List<ECKey> getPubKeys() {
        if (!this.isSentToMultiSig()) {
            throw new ScriptException("Only usable for multisig scripts.");
        }
        ArrayList result = Lists.newArrayList();
        int numKeys = Script.decodeFromOpN(this.chunks.get((int)(this.chunks.size() - 2)).opcode);
        for (int i = 0; i < numKeys; ++i) {
            result.add(ECKey.fromPublicOnly((byte[])this.chunks.get((int)(1 + i)).data));
        }
        return result;
    }

    private int findSigInRedeem(byte[] signatureBytes, Sha256Hash hash) {
        Preconditions.checkArgument((boolean)this.chunks.get(0).isOpCode());
        int numKeys = Script.decodeFromOpN(this.chunks.get((int)(this.chunks.size() - 2)).opcode);
        throw new IllegalStateException("Could not find matching key for signature on " + hash.toString() + " sig " + HexUtil.encode((byte[])signatureBytes));
    }

    private static int getSigOpCount(List<ScriptChunk> chunks, boolean accurate) throws ScriptException {
        int sigOps = 0;
        int lastOpCode = 255;
        for (ScriptChunk chunk : chunks) {
            if (!chunk.isOpCode()) continue;
            switch (chunk.opcode) {
                case 172: 
                case 173: {
                    ++sigOps;
                    break;
                }
                case 174: 
                case 175: {
                    if (accurate && lastOpCode >= 81 && lastOpCode <= 96) {
                        sigOps += Script.decodeFromOpN(lastOpCode);
                        break;
                    }
                    sigOps += 20;
                    break;
                }
            }
            lastOpCode = chunk.opcode;
        }
        return sigOps;
    }

    public static int decodeFromOpN(int opcode) {
        Preconditions.checkArgument((opcode == 0 || opcode == 79 || opcode >= 81 && opcode <= 96 ? 1 : 0) != 0, (Object)"decodeFromOpN called on non OP_N opcode");
        if (opcode == 0) {
            return 0;
        }
        if (opcode == 79) {
            return -1;
        }
        return opcode + 1 - 81;
    }

    static int encodeToOpN(int value) {
        Preconditions.checkArgument((value >= -1 && value <= 16 ? 1 : 0) != 0, (Object)("encodeToOpN called for " + value + " which we cannot encode in an opcode."));
        if (value == 0) {
            return 0;
        }
        if (value == -1) {
            return 79;
        }
        return value - 1 + 81;
    }

    public static int getSigOpCount(byte[] program) throws ScriptException {
        Script script = new Script();
        try {
            script.parse(program);
        }
        catch (ScriptException scriptException) {
            // empty catch block
        }
        return Script.getSigOpCount(script.chunks, false);
    }

    public static long getP2SHSigOpCount(byte[] scriptSig) throws ScriptException {
        Script script = new Script();
        try {
            script.parse(scriptSig);
        }
        catch (ScriptException scriptException) {
            // empty catch block
        }
        for (int i = script.chunks.size() - 1; i >= 0; --i) {
            if (script.chunks.get(i).isOpCode()) continue;
            Script subScript = new Script();
            subScript.parse(script.chunks.get((int)i).data);
            return Script.getSigOpCount(subScript.chunks, true);
        }
        return 0L;
    }

    public int getNumberOfSignaturesRequiredToSpend() {
        if (this.isSentToMultiSig()) {
            ScriptChunk nChunk = this.chunks.get(0);
            return Script.decodeFromOpN(nChunk.opcode);
        }
        if (this.isSentToAddress() || this.isSentToRawPubKey()) {
            return 1;
        }
        if (this.isPayToScriptHash()) {
            throw new IllegalStateException("For P2SH number of signatures depends on redeem script");
        }
        throw new IllegalStateException("Unsupported script type");
    }

    public int getNumberOfBytesRequiredToSpend(@Nullable ECKey pubKey, @Nullable Script redeemScript) {
        if (this.isPayToScriptHash()) {
            Preconditions.checkArgument((redeemScript != null ? 1 : 0) != 0, (Object)"P2SH script requires redeemScript to be spent");
            return redeemScript.getNumberOfSignaturesRequiredToSpend() * 75 + redeemScript.getProgram().length;
        }
        if (this.isSentToMultiSig()) {
            return this.getNumberOfSignaturesRequiredToSpend() * 75 + 1;
        }
        if (this.isSentToRawPubKey()) {
            return 75;
        }
        if (this.isSentToAddress()) {
            int uncompressedPubKeySize = 65;
            return 75 + (pubKey != null ? pubKey.getPubKey().length : uncompressedPubKeySize);
        }
        throw new IllegalStateException("Unsupported script type");
    }

    public boolean isPayToScriptHash() {
        byte[] program = this.getProgram();
        return program.length == 26 && (program[0] & 0xFF) == 169 && (program[1] & 0xFF) == 23 && (program[25] & 0xFF) == 135;
    }

    public boolean isSentToMultiSig() {
        if (this.chunks.size() < 4) {
            return false;
        }
        ScriptChunk chunk = this.chunks.get(this.chunks.size() - 1);
        if (!chunk.isOpCode()) {
            return false;
        }
        if (!chunk.equalsOpCode(174) && !chunk.equalsOpCode(175)) {
            return false;
        }
        try {
            ScriptChunk m = this.chunks.get(this.chunks.size() - 2);
            if (!m.isOpCode()) {
                return false;
            }
            int numKeys = Script.decodeFromOpN(m.opcode);
            if (numKeys < 1 || this.chunks.size() != 3 + numKeys) {
                return false;
            }
            for (int i = 1; i < this.chunks.size() - 2; ++i) {
                if (!this.chunks.get(i).isOpCode()) continue;
                return false;
            }
            if (Script.decodeFromOpN(this.chunks.get((int)0).opcode) < 1) {
                return false;
            }
        }
        catch (IllegalArgumentException e) {
            return false;
        }
        return true;
    }

    public boolean isSentToCLTVPaymentChannel() {
        if (this.chunks.size() != 10) {
            return false;
        }
        if (!this.chunks.get(0).equalsOpCode(99)) {
            return false;
        }
        if (!this.chunks.get(2).equalsOpCode(173)) {
            return false;
        }
        if (!this.chunks.get(3).equalsOpCode(103)) {
            return false;
        }
        if (!this.chunks.get(5).equalsOpCode(177)) {
            return false;
        }
        if (!this.chunks.get(6).equalsOpCode(117)) {
            return false;
        }
        if (!this.chunks.get(7).equalsOpCode(104)) {
            return false;
        }
        return this.chunks.get(9).equalsOpCode(172);
    }

    private static boolean equalsRange(byte[] a, int start, byte[] b) {
        if (start + b.length > a.length) {
            return false;
        }
        for (int i = 0; i < b.length; ++i) {
            if (a[i + start] == b[i]) continue;
            return false;
        }
        return true;
    }

    public static byte[] removeAllInstancesOf(byte[] inputScript, byte[] chunkToRemove) {
        int additionalBytes;
        UnsafeByteArrayOutputStream bos = new UnsafeByteArrayOutputStream(inputScript.length);
        for (int cursor = 0; cursor < inputScript.length; cursor += additionalBytes) {
            boolean skip = Script.equalsRange(inputScript, cursor, chunkToRemove);
            int opcode = inputScript[cursor++] & 0xFF;
            additionalBytes = 0;
            if (opcode >= 0 && opcode < 76) {
                additionalBytes = opcode;
            } else if (opcode == 76) {
                additionalBytes = (0xFF & inputScript[cursor]) + 1;
            } else if (opcode == 77) {
                additionalBytes = (0xFF & inputScript[cursor] | (0xFF & inputScript[cursor + 1]) << 8) + 2;
            } else if (opcode == 78) {
                additionalBytes = (0xFF & inputScript[cursor] | (0xFF & inputScript[cursor + 1]) << 8 | (0xFF & inputScript[cursor + 1]) << 16 | (0xFF & inputScript[cursor + 1]) << 24) + 4;
            }
            if (skip) continue;
            try {
                bos.write(opcode);
                bos.write(Arrays.copyOfRange(inputScript, cursor, cursor + additionalBytes));
                continue;
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
        return bos.toByteArray();
    }

    public static byte[] removeAllInstancesOfOp(byte[] inputScript, int opCode) {
        return Script.removeAllInstancesOf(inputScript, new byte[]{(byte)opCode});
    }

    private static boolean castToBool(byte[] data) {
        for (int i = 0; i < data.length; ++i) {
            if (data[i] == 0) continue;
            return i != data.length - 1 || (data[i] & 0xFF) != 128;
        }
        return false;
    }

    private static BigInteger castToBigInteger(byte[] chunk) throws ScriptException {
        if (chunk.length > 4) {
            throw new ScriptException("Script attempted to use an integer larger than 4 bytes");
        }
        return SerializeUtils.decodeMPI((byte[])ByteUtils.reverseBytes((byte[])chunk), (boolean)false);
    }

    private static BigInteger castToBigInteger(byte[] chunk, int maxLength) throws ScriptException {
        if (chunk.length > maxLength) {
            throw new ScriptException("Script attempted to use an integer larger than " + maxLength + " bytes");
        }
        return SerializeUtils.decodeMPI((byte[])ByteUtils.reverseBytes((byte[])chunk), (boolean)false);
    }

    public boolean isOpReturn() {
        return this.chunks.size() > 0 && this.chunks.get(0).equalsOpCode(106);
    }

    @Deprecated
    public static void executeScript(@Nullable Transaction txContainingThis, long index, Script script, LinkedList<byte[]> stack, boolean enforceNullDummy) throws ScriptException {
        EnumSet<VerifyFlag> flags = enforceNullDummy ? EnumSet.of(VerifyFlag.NULLDUMMY) : EnumSet.noneOf(VerifyFlag.class);
        Script.executeScript(txContainingThis, index, script, stack, flags);
    }

    public static void executeScript(@Nullable Transaction txContainingThis, long index, Script script, LinkedList<byte[]> stack, Set<VerifyFlag> verifyFlags) throws ScriptException {
        int opCount = 0;
        int lastCodeSepLocation = 0;
        LinkedList<byte[]> altstack = new LinkedList<byte[]>();
        LinkedList<Boolean> ifStack = new LinkedList<Boolean>();
        block75: for (ScriptChunk chunk : script.chunks) {
            boolean shouldExecute;
            boolean bl = shouldExecute = !ifStack.contains(false);
            if (chunk.opcode == 0) {
                if (!shouldExecute) continue;
                stack.add(new byte[0]);
            } else if (!chunk.isOpCode()) {
                if ((long)chunk.data.length > 520L) {
                    throw new ScriptException("Attempted to push a entity string larger than 520 bytes");
                }
                if (!shouldExecute) continue;
                stack.add(chunk.data);
            } else {
                int opcode = chunk.opcode;
                if (opcode > 96 && ++opCount > 201) {
                    throw new ScriptException("More script operations than is allowed");
                }
                if (opcode == 101 || opcode == 102) {
                    throw new ScriptException("Script included OP_VERIF or OP_VERNOTIF");
                }
                if (opcode == 126 || opcode == 127 || opcode == 128 || opcode == 129 || opcode == 131 || opcode == 132 || opcode == 133 || opcode == 134 || opcode == 141 || opcode == 142 || opcode == 149 || opcode == 150 || opcode == 151 || opcode == 152 || opcode == 153) {
                    throw new ScriptException("Script included a disabled Script Op.");
                }
                switch (opcode) {
                    case 99: {
                        if (!shouldExecute) {
                            ifStack.add(false);
                            continue block75;
                        }
                        if (stack.size() < 1) {
                            throw new ScriptException("Attempted OP_IF on an empty stack");
                        }
                        ifStack.add(Script.castToBool(stack.pollLast()));
                        continue block75;
                    }
                    case 100: {
                        if (!shouldExecute) {
                            ifStack.add(false);
                            continue block75;
                        }
                        if (stack.size() < 1) {
                            throw new ScriptException("Attempted OP_NOTIF on an empty stack");
                        }
                        ifStack.add(!Script.castToBool(stack.pollLast()));
                        continue block75;
                    }
                    case 103: {
                        if (ifStack.isEmpty()) {
                            throw new ScriptException("Attempted OP_ELSE without OP_IF/NOTIF");
                        }
                        ifStack.add((Boolean)ifStack.pollLast() == false);
                        continue block75;
                    }
                    case 104: {
                        if (ifStack.isEmpty()) {
                            throw new ScriptException("Attempted OP_ENDIF without OP_IF/NOTIF");
                        }
                        ifStack.pollLast();
                        continue block75;
                    }
                }
                if (!shouldExecute) continue;
                switch (opcode) {
                    case 79: {
                        stack.add(ByteUtils.reverseBytes((byte[])SerializeUtils.encodeMPI((BigInteger)BigInteger.ONE.negate(), (boolean)false)));
                        break;
                    }
                    case 81: 
                    case 82: 
                    case 83: 
                    case 84: 
                    case 85: 
                    case 86: 
                    case 87: 
                    case 88: 
                    case 89: 
                    case 90: 
                    case 91: 
                    case 92: 
                    case 93: 
                    case 94: 
                    case 95: 
                    case 96: {
                        stack.add(ByteUtils.reverseBytes((byte[])SerializeUtils.encodeMPI((BigInteger)BigInteger.valueOf(Script.decodeFromOpN(opcode)), (boolean)false)));
                        break;
                    }
                    case 97: {
                        break;
                    }
                    case 105: {
                        if (stack.size() < 1) {
                            throw new ScriptException("Attempted OP_VERIFY on an empty stack");
                        }
                        if (Script.castToBool(stack.pollLast())) break;
                        throw new ScriptException("OP_VERIFY failed");
                    }
                    case 106: {
                        throw new ScriptException("Script called OP_RETURN");
                    }
                    case 107: {
                        if (stack.size() < 1) {
                            throw new ScriptException("Attempted OP_TOALTSTACK on an empty stack");
                        }
                        altstack.add(stack.pollLast());
                        break;
                    }
                    case 108: {
                        if (altstack.size() < 1) {
                            throw new ScriptException("Attempted OP_FROMALTSTACK on an empty altstack");
                        }
                        stack.add((byte[])altstack.pollLast());
                        break;
                    }
                    case 109: {
                        if (stack.size() < 2) {
                            throw new ScriptException("Attempted OP_2DROP on a stack with size < 2");
                        }
                        stack.pollLast();
                        stack.pollLast();
                        break;
                    }
                    case 110: {
                        if (stack.size() < 2) {
                            throw new ScriptException("Attempted OP_2DUP on a stack with size < 2");
                        }
                        Iterator<byte[]> it2DUP = stack.descendingIterator();
                        byte[] OP2DUPtmpChunk2 = it2DUP.next();
                        stack.add(it2DUP.next());
                        stack.add(OP2DUPtmpChunk2);
                        break;
                    }
                    case 111: {
                        if (stack.size() < 3) {
                            throw new ScriptException("Attempted OP_3DUP on a stack with size < 3");
                        }
                        Iterator<byte[]> it3DUP = stack.descendingIterator();
                        byte[] OP3DUPtmpChunk3 = it3DUP.next();
                        byte[] OP3DUPtmpChunk2 = it3DUP.next();
                        stack.add(it3DUP.next());
                        stack.add(OP3DUPtmpChunk2);
                        stack.add(OP3DUPtmpChunk3);
                        break;
                    }
                    case 112: {
                        if (stack.size() < 4) {
                            throw new ScriptException("Attempted OP_2OVER on a stack with size < 4");
                        }
                        Iterator<byte[]> it2OVER = stack.descendingIterator();
                        it2OVER.next();
                        it2OVER.next();
                        byte[] OP2OVERtmpChunk2 = it2OVER.next();
                        stack.add(it2OVER.next());
                        stack.add(OP2OVERtmpChunk2);
                        break;
                    }
                    case 113: {
                        if (stack.size() < 6) {
                            throw new ScriptException("Attempted OP_2ROT on a stack with size < 6");
                        }
                        byte[] OP2ROTtmpChunk6 = stack.pollLast();
                        byte[] OP2ROTtmpChunk5 = stack.pollLast();
                        byte[] OP2ROTtmpChunk4 = stack.pollLast();
                        byte[] OP2ROTtmpChunk3 = stack.pollLast();
                        byte[] OP2ROTtmpChunk2 = stack.pollLast();
                        byte[] OP2ROTtmpChunk1 = stack.pollLast();
                        stack.add(OP2ROTtmpChunk3);
                        stack.add(OP2ROTtmpChunk4);
                        stack.add(OP2ROTtmpChunk5);
                        stack.add(OP2ROTtmpChunk6);
                        stack.add(OP2ROTtmpChunk1);
                        stack.add(OP2ROTtmpChunk2);
                        break;
                    }
                    case 114: {
                        if (stack.size() < 4) {
                            throw new ScriptException("Attempted OP_2SWAP on a stack with size < 4");
                        }
                        byte[] OP2SWAPtmpChunk4 = stack.pollLast();
                        byte[] OP2SWAPtmpChunk3 = stack.pollLast();
                        byte[] OP2SWAPtmpChunk2 = stack.pollLast();
                        byte[] OP2SWAPtmpChunk1 = stack.pollLast();
                        stack.add(OP2SWAPtmpChunk3);
                        stack.add(OP2SWAPtmpChunk4);
                        stack.add(OP2SWAPtmpChunk1);
                        stack.add(OP2SWAPtmpChunk2);
                        break;
                    }
                    case 115: {
                        if (stack.size() < 1) {
                            throw new ScriptException("Attempted OP_IFDUP on an empty stack");
                        }
                        if (!Script.castToBool(stack.getLast())) break;
                        stack.add(stack.getLast());
                        break;
                    }
                    case 116: {
                        stack.add(ByteUtils.reverseBytes((byte[])SerializeUtils.encodeMPI((BigInteger)BigInteger.valueOf(stack.size()), (boolean)false)));
                        break;
                    }
                    case 117: {
                        if (stack.size() < 1) {
                            throw new ScriptException("Attempted OP_DROP on an empty stack");
                        }
                        stack.pollLast();
                        break;
                    }
                    case 118: {
                        if (stack.size() < 1) {
                            throw new ScriptException("Attempted OP_DUP on an empty stack");
                        }
                        stack.add(stack.getLast());
                        break;
                    }
                    case 119: {
                        if (stack.size() < 2) {
                            throw new ScriptException("Attempted OP_NIP on a stack with size < 2");
                        }
                        byte[] OPNIPtmpChunk = stack.pollLast();
                        stack.pollLast();
                        stack.add(OPNIPtmpChunk);
                        break;
                    }
                    case 120: {
                        if (stack.size() < 2) {
                            throw new ScriptException("Attempted OP_OVER on a stack with size < 2");
                        }
                        Iterator<byte[]> itOVER = stack.descendingIterator();
                        itOVER.next();
                        stack.add(itOVER.next());
                        break;
                    }
                    case 121: 
                    case 122: {
                        if (stack.size() < 1) {
                            throw new ScriptException("Attempted OP_PICK/OP_ROLL on an empty stack");
                        }
                        long val = Script.castToBigInteger(stack.pollLast()).longValue();
                        if (val < 0L || val >= (long)stack.size()) {
                            throw new ScriptException("OP_PICK/OP_ROLL attempted to get entity deeper than stack size");
                        }
                        Iterator<byte[]> itPICK = stack.descendingIterator();
                        for (long i = 0L; i < val; ++i) {
                            itPICK.next();
                        }
                        byte[] OPROLLtmpChunk = itPICK.next();
                        if (opcode == 122) {
                            itPICK.remove();
                        }
                        stack.add(OPROLLtmpChunk);
                        break;
                    }
                    case 123: {
                        if (stack.size() < 3) {
                            throw new ScriptException("Attempted OP_ROT on a stack with size < 3");
                        }
                        byte[] OPROTtmpChunk3 = stack.pollLast();
                        byte[] OPROTtmpChunk2 = stack.pollLast();
                        byte[] OPROTtmpChunk1 = stack.pollLast();
                        stack.add(OPROTtmpChunk2);
                        stack.add(OPROTtmpChunk3);
                        stack.add(OPROTtmpChunk1);
                        break;
                    }
                    case 124: 
                    case 125: {
                        if (stack.size() < 2) {
                            throw new ScriptException("Attempted OP_SWAP on a stack with size < 2");
                        }
                        byte[] OPSWAPtmpChunk2 = stack.pollLast();
                        byte[] OPSWAPtmpChunk1 = stack.pollLast();
                        stack.add(OPSWAPtmpChunk2);
                        stack.add(OPSWAPtmpChunk1);
                        if (opcode != 125) break;
                        stack.add(OPSWAPtmpChunk2);
                        break;
                    }
                    case 126: 
                    case 127: 
                    case 128: 
                    case 129: {
                        throw new ScriptException("Attempted to use disabled Script Op.");
                    }
                    case 130: {
                        if (stack.size() < 1) {
                            throw new ScriptException("Attempted OP_SIZE on an empty stack");
                        }
                        stack.add(ByteUtils.reverseBytes((byte[])SerializeUtils.encodeMPI((BigInteger)BigInteger.valueOf(stack.getLast().length), (boolean)false)));
                        break;
                    }
                    case 131: 
                    case 132: 
                    case 133: 
                    case 134: {
                        throw new ScriptException("Attempted to use disabled Script Op.");
                    }
                    case 135: {
                        byte[] byArray;
                        if (stack.size() < 2) {
                            throw new ScriptException("Attempted OP_EQUAL on a stack with size < 2");
                        }
                        if (AddressTool.checkPublicKeyHash(stack.pollLast(), stack.pollLast())) {
                            byte[] byArray2 = new byte[1];
                            byArray = byArray2;
                            byArray2[0] = 1;
                        } else {
                            byArray = new byte[]{};
                        }
                        stack.add(byArray);
                        break;
                    }
                    case 136: {
                        if (stack.size() < 2) {
                            throw new ScriptException("Attempted OP_EQUALVERIFY on a stack with size < 2");
                        }
                        if (AddressTool.checkPublicKeyHash(stack.pollLast(), stack.pollLast())) break;
                        throw new ScriptException("OP_EQUALVERIFY: non-equal entity");
                    }
                    case 139: 
                    case 140: 
                    case 143: 
                    case 144: 
                    case 145: 
                    case 146: {
                        if (stack.size() < 1) {
                            throw new ScriptException("Attempted a numeric op on an empty stack");
                        }
                        BigInteger numericOPnum = Script.castToBigInteger(stack.pollLast());
                        switch (opcode) {
                            case 139: {
                                numericOPnum = numericOPnum.add(BigInteger.ONE);
                                break;
                            }
                            case 140: {
                                numericOPnum = numericOPnum.subtract(BigInteger.ONE);
                                break;
                            }
                            case 143: {
                                numericOPnum = numericOPnum.negate();
                                break;
                            }
                            case 144: {
                                if (numericOPnum.signum() >= 0) break;
                                numericOPnum = numericOPnum.negate();
                                break;
                            }
                            case 145: {
                                if (numericOPnum.equals(BigInteger.ZERO)) {
                                    numericOPnum = BigInteger.ONE;
                                    break;
                                }
                                numericOPnum = BigInteger.ZERO;
                                break;
                            }
                            case 146: {
                                if (numericOPnum.equals(BigInteger.ZERO)) {
                                    numericOPnum = BigInteger.ZERO;
                                    break;
                                }
                                numericOPnum = BigInteger.ONE;
                                break;
                            }
                            default: {
                                throw new AssertionError((Object)"Unreachable");
                            }
                        }
                        stack.add(ByteUtils.reverseBytes((byte[])SerializeUtils.encodeMPI((BigInteger)numericOPnum, (boolean)false)));
                        break;
                    }
                    case 141: 
                    case 142: {
                        throw new ScriptException("Attempted to use disabled Script Op.");
                    }
                    case 147: 
                    case 148: 
                    case 154: 
                    case 155: 
                    case 156: 
                    case 158: 
                    case 159: 
                    case 160: 
                    case 161: 
                    case 162: 
                    case 163: 
                    case 164: {
                        BigInteger numericOPresult;
                        if (stack.size() < 2) {
                            throw new ScriptException("Attempted a numeric op on a stack with size < 2");
                        }
                        BigInteger numericOPnum2 = Script.castToBigInteger(stack.pollLast());
                        BigInteger numericOPnum1 = Script.castToBigInteger(stack.pollLast());
                        switch (opcode) {
                            case 147: {
                                numericOPresult = numericOPnum1.add(numericOPnum2);
                                break;
                            }
                            case 148: {
                                numericOPresult = numericOPnum1.subtract(numericOPnum2);
                                break;
                            }
                            case 154: {
                                if (!numericOPnum1.equals(BigInteger.ZERO) && !numericOPnum2.equals(BigInteger.ZERO)) {
                                    numericOPresult = BigInteger.ONE;
                                    break;
                                }
                                numericOPresult = BigInteger.ZERO;
                                break;
                            }
                            case 155: {
                                if (!numericOPnum1.equals(BigInteger.ZERO) || !numericOPnum2.equals(BigInteger.ZERO)) {
                                    numericOPresult = BigInteger.ONE;
                                    break;
                                }
                                numericOPresult = BigInteger.ZERO;
                                break;
                            }
                            case 156: {
                                if (numericOPnum1.equals(numericOPnum2)) {
                                    numericOPresult = BigInteger.ONE;
                                    break;
                                }
                                numericOPresult = BigInteger.ZERO;
                                break;
                            }
                            case 158: {
                                if (!numericOPnum1.equals(numericOPnum2)) {
                                    numericOPresult = BigInteger.ONE;
                                    break;
                                }
                                numericOPresult = BigInteger.ZERO;
                                break;
                            }
                            case 159: {
                                if (numericOPnum1.compareTo(numericOPnum2) < 0) {
                                    numericOPresult = BigInteger.ONE;
                                    break;
                                }
                                numericOPresult = BigInteger.ZERO;
                                break;
                            }
                            case 160: {
                                if (numericOPnum1.compareTo(numericOPnum2) > 0) {
                                    numericOPresult = BigInteger.ONE;
                                    break;
                                }
                                numericOPresult = BigInteger.ZERO;
                                break;
                            }
                            case 161: {
                                if (numericOPnum1.compareTo(numericOPnum2) <= 0) {
                                    numericOPresult = BigInteger.ONE;
                                    break;
                                }
                                numericOPresult = BigInteger.ZERO;
                                break;
                            }
                            case 162: {
                                if (numericOPnum1.compareTo(numericOPnum2) >= 0) {
                                    numericOPresult = BigInteger.ONE;
                                    break;
                                }
                                numericOPresult = BigInteger.ZERO;
                                break;
                            }
                            case 163: {
                                if (numericOPnum1.compareTo(numericOPnum2) < 0) {
                                    numericOPresult = numericOPnum1;
                                    break;
                                }
                                numericOPresult = numericOPnum2;
                                break;
                            }
                            case 164: {
                                if (numericOPnum1.compareTo(numericOPnum2) > 0) {
                                    numericOPresult = numericOPnum1;
                                    break;
                                }
                                numericOPresult = numericOPnum2;
                                break;
                            }
                            default: {
                                throw new RuntimeException("Opcode switched at runtime?");
                            }
                        }
                        stack.add(ByteUtils.reverseBytes((byte[])SerializeUtils.encodeMPI((BigInteger)numericOPresult, (boolean)false)));
                        break;
                    }
                    case 149: 
                    case 150: 
                    case 151: 
                    case 152: 
                    case 153: {
                        throw new ScriptException("Attempted to use disabled Script Op.");
                    }
                    case 157: {
                        if (stack.size() < 2) {
                            throw new ScriptException("Attempted OP_NUMEQUALVERIFY on a stack with size < 2");
                        }
                        BigInteger OPNUMEQUALVERIFYnum2 = Script.castToBigInteger(stack.pollLast());
                        BigInteger OPNUMEQUALVERIFYnum1 = Script.castToBigInteger(stack.pollLast());
                        if (OPNUMEQUALVERIFYnum1.equals(OPNUMEQUALVERIFYnum2)) break;
                        throw new ScriptException("OP_NUMEQUALVERIFY failed");
                    }
                    case 165: {
                        BigInteger OPWITHINnum1;
                        if (stack.size() < 3) {
                            throw new ScriptException("Attempted OP_WITHIN on a stack with size < 3");
                        }
                        BigInteger OPWITHINnum3 = Script.castToBigInteger(stack.pollLast());
                        BigInteger OPWITHINnum2 = Script.castToBigInteger(stack.pollLast());
                        if (OPWITHINnum2.compareTo(OPWITHINnum1 = Script.castToBigInteger(stack.pollLast())) <= 0 && OPWITHINnum1.compareTo(OPWITHINnum3) < 0) {
                            stack.add(ByteUtils.reverseBytes((byte[])SerializeUtils.encodeMPI((BigInteger)BigInteger.ONE, (boolean)false)));
                            break;
                        }
                        stack.add(ByteUtils.reverseBytes((byte[])SerializeUtils.encodeMPI((BigInteger)BigInteger.ZERO, (boolean)false)));
                        break;
                    }
                    case 166: {
                        if (stack.size() < 1) {
                            throw new ScriptException("Attempted OP_RIPEMD160 on an empty stack");
                        }
                        RIPEMD160Digest digest = new RIPEMD160Digest();
                        byte[] dataToHash = stack.pollLast();
                        digest.update(dataToHash, 0, dataToHash.length);
                        byte[] ripmemdHash = new byte[20];
                        digest.doFinal(ripmemdHash, 0);
                        stack.add(ripmemdHash);
                        break;
                    }
                    case 167: {
                        if (stack.size() < 1) {
                            throw new ScriptException("Attempted OP_SHA1 on an empty stack");
                        }
                        try {
                            stack.add(MessageDigest.getInstance("SHA-1").digest(stack.pollLast()));
                            break;
                        }
                        catch (NoSuchAlgorithmException e) {
                            throw new RuntimeException(e);
                        }
                    }
                    case 168: {
                        if (stack.size() < 1) {
                            throw new ScriptException("Attempted OP_SHA256 on an empty stack");
                        }
                        stack.add(Sha256Hash.hash((byte[])stack.pollLast()));
                        break;
                    }
                    case 169: {
                        if (stack.size() < 1) {
                            throw new ScriptException("Attempted OP_HASH160 on an empty stack");
                        }
                        stack.add(SerializeUtils.sha256hash160((byte[])stack.pollLast()));
                        break;
                    }
                    case 170: {
                        if (stack.size() < 1) {
                            throw new ScriptException("Attempted OP_SHA256 on an empty stack");
                        }
                        stack.add(Sha256Hash.hashTwice((byte[])stack.pollLast()));
                        break;
                    }
                    case 171: {
                        lastCodeSepLocation = chunk.getStartLocationInProgram() + 1;
                        break;
                    }
                    case 172: 
                    case 173: {
                        if (txContainingThis == null) {
                            throw new IllegalStateException("Script attempted signature check but no tx was provided");
                        }
                        Script.executeCheckSig(txContainingThis, (int)index, script, stack, lastCodeSepLocation, opcode, verifyFlags);
                        break;
                    }
                    case 174: 
                    case 175: {
                        if (txContainingThis == null) {
                            throw new IllegalStateException("Script attempted signature check but no tx was provided");
                        }
                        opCount = Script.executeMultiSig(txContainingThis, (int)index, script, stack, opCount, lastCodeSepLocation, opcode, verifyFlags);
                        break;
                    }
                    case 177: {
                        if (!verifyFlags.contains((Object)VerifyFlag.CHECKLOCKTIMEVERIFY)) {
                            if (!verifyFlags.contains((Object)VerifyFlag.DISCOURAGE_UPGRADABLE_NOPS)) break;
                            throw new ScriptException("Script used a reserved opcode " + opcode);
                        }
                        Script.executeCheckLockTimeVerify(txContainingThis, (int)index, script, stack, lastCodeSepLocation, opcode, verifyFlags);
                        break;
                    }
                    case 176: 
                    case 178: 
                    case 179: 
                    case 180: 
                    case 181: 
                    case 182: 
                    case 183: 
                    case 184: 
                    case 185: {
                        if (!verifyFlags.contains((Object)VerifyFlag.DISCOURAGE_UPGRADABLE_NOPS)) break;
                        throw new ScriptException("Script used a reserved opcode " + opcode);
                    }
                    default: {
                        throw new ScriptException("Script used a reserved opcode " + opcode);
                    }
                }
            }
            if (stack.size() + altstack.size() <= 1000 && stack.size() + altstack.size() >= 0) continue;
            throw new ScriptException("Stack size exceeded range");
        }
        if (!ifStack.isEmpty()) {
            throw new ScriptException("OP_IF/OP_NOTIF without OP_ENDIF");
        }
    }

    private static void executeCheckLockTimeVerify(Transaction txContainingThis, int index, Script script, LinkedList<byte[]> stack, int lastCodeSepLocation, int opcode, Set<VerifyFlag> verifyFlags) throws ScriptException {
        if (stack.size() < 1) {
            throw new ScriptException("Attempted OP_CHECKLOCKTIMEVERIFY on a stack with size < 1");
        }
        BigInteger nLockTime = Script.castToBigInteger(stack.getLast(), 5);
        if (nLockTime.compareTo(BigInteger.ZERO) < 0) {
            throw new ScriptException("Negative locktime");
        }
        throw new ScriptException("Locktime requirement type mismatch");
    }

    private static void executeCheckSig(Transaction txContainingThis, int index, Script script, LinkedList<byte[]> stack, int lastCodeSepLocation, int opcode, Set<VerifyFlag> verifyFlags) throws ScriptException {
        boolean sigValid;
        block10: {
            boolean requireCanonical;
            boolean bl = requireCanonical = verifyFlags.contains((Object)VerifyFlag.STRICTENC) || verifyFlags.contains((Object)VerifyFlag.DERSIG) || verifyFlags.contains((Object)VerifyFlag.LOW_S);
            if (stack.size() < 2) {
                throw new ScriptException("Attempted OP_CHECKSIG(VERIFY) on a stack with size < 2");
            }
            byte[] pubKey = stack.pollLast();
            byte[] sigBytes = stack.pollLast();
            byte[] prog = script.getProgram();
            byte[] connectedScript = Arrays.copyOfRange(prog, lastCodeSepLocation, prog.length);
            UnsafeByteArrayOutputStream outStream = new UnsafeByteArrayOutputStream(sigBytes.length + 1);
            try {
                Script.writeBytes((OutputStream)outStream, sigBytes);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
            connectedScript = Script.removeAllInstancesOf(connectedScript, outStream.toByteArray());
            sigValid = false;
            try {
                sigValid = Script.verifySign(txContainingThis.getHash().getBytes(), sigBytes, pubKey);
            }
            catch (Exception e1) {
                if (e1.getMessage().contains("Reached past end of ASN.1 stream")) break block10;
                log.warn("Signature checking failed!", (Throwable)e1);
            }
        }
        if (opcode == 172) {
            byte[] byArray;
            if (sigValid) {
                byte[] byArray2 = new byte[1];
                byArray = byArray2;
                byArray2[0] = 1;
            } else {
                byArray = new byte[]{};
            }
            stack.add(byArray);
        } else if (opcode == 173 && !sigValid) {
            throw new ScriptException("Script failed OP_CHECKSIGVERIFY");
        }
    }

    private static int executeMultiSig(Transaction txContainingThis, int index, Script script, LinkedList<byte[]> stack, int opCount, int lastCodeSepLocation, int opcode, Set<VerifyFlag> verifyFlags) throws ScriptException {
        boolean requireCanonical;
        boolean bl = requireCanonical = verifyFlags.contains((Object)VerifyFlag.STRICTENC) || verifyFlags.contains((Object)VerifyFlag.DERSIG) || verifyFlags.contains((Object)VerifyFlag.LOW_S);
        if (stack.size() < 2) {
            throw new ScriptException("Attempted OP_CHECKMULTISIG(VERIFY) on a stack with size < 2");
        }
        int pubKeyCount = Script.castToBigInteger(stack.pollLast()).intValue();
        if (pubKeyCount < 0 || pubKeyCount > 20) {
            throw new ScriptException("OP_CHECKMULTISIG(VERIFY) with pubkey count out of range");
        }
        if ((opCount += pubKeyCount) > 201) {
            throw new ScriptException("Total op count > 201 during OP_CHECKMULTISIG(VERIFY)");
        }
        if (stack.size() < pubKeyCount + 1) {
            throw new ScriptException("Attempted OP_CHECKMULTISIG(VERIFY) on a stack with size < num_of_pubkeys + 2");
        }
        LinkedList<byte[]> pubkeys = new LinkedList<byte[]>();
        for (int i = 0; i < pubKeyCount; ++i) {
            byte[] pubKey = stack.pollLast();
            pubkeys.add(pubKey);
        }
        int sigCount = Script.castToBigInteger(stack.pollLast()).intValue();
        if (sigCount < 0 || sigCount > pubKeyCount) {
            throw new ScriptException("OP_CHECKMULTISIG(VERIFY) with sig count out of range");
        }
        if (stack.size() < sigCount + 1) {
            throw new ScriptException("Attempted OP_CHECKMULTISIG(VERIFY) on a stack with size < num_of_pubkeys + num_of_signatures + 3");
        }
        LinkedList<byte[]> sigs = new LinkedList<byte[]>();
        for (int i = 0; i < sigCount; ++i) {
            byte[] sig = stack.pollLast();
            sigs.add(sig);
        }
        byte[] prog = script.getProgram();
        byte[] connectedScript = Arrays.copyOfRange(prog, lastCodeSepLocation, prog.length);
        for (byte[] sig : sigs) {
            UnsafeByteArrayOutputStream outStream = new UnsafeByteArrayOutputStream(sig.length + 1);
            try {
                Script.writeBytes((OutputStream)outStream, sig);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
            connectedScript = Script.removeAllInstancesOf(connectedScript, outStream.toByteArray());
        }
        boolean valid = true;
        while (sigs.size() > 0) {
            byte[] pubKey = (byte[])pubkeys.pollFirst();
            try {
                if (ECKey.verify((byte[])txContainingThis.getHash().getBytes(), (byte[])((byte[])sigs.getFirst()), (byte[])pubKey)) {
                    sigs.pollFirst();
                }
            }
            catch (Exception e) {
                e.printStackTrace();
            }
            if (sigs.size() <= pubkeys.size()) continue;
            valid = false;
            break;
        }
        byte[] nullDummy = stack.pollLast();
        if (verifyFlags.contains((Object)VerifyFlag.NULLDUMMY) && nullDummy.length > 0) {
            throw new ScriptException("OP_CHECKMULTISIG(VERIFY) with non-null nulldummy: " + Arrays.toString(nullDummy));
        }
        if (opcode == 174) {
            byte[] byArray;
            if (valid) {
                byte[] byArray2 = new byte[1];
                byArray = byArray2;
                byArray2[0] = 1;
            } else {
                byArray = new byte[]{};
            }
            stack.add(byArray);
        } else if (opcode == 175 && !valid) {
            throw new ScriptException("Script failed OP_CHECKMULTISIGVERIFY");
        }
        return opCount;
    }

    @Deprecated
    public boolean correctlySpends(Transaction txContainingThis, long scriptSigIndex, Script scriptPubKey) {
        return this.correctlySpends(txContainingThis, scriptSigIndex, scriptPubKey, ALL_VERIFY_FLAGS);
    }

    public boolean correctlyNulsSpends(Transaction txContainingThis, long scriptSigIndex, Script scriptPubKey) {
        return this.correctlySpends(txContainingThis, scriptSigIndex, scriptPubKey, ALL_VERIFY_FLAGS);
    }

    public boolean correctlySpends(Transaction txContainingThis, long scriptSigIndex, Script scriptPubKey, Set<VerifyFlag> verifyFlags) {
        try {
            if (verifyFlags == null) {
                verifyFlags = ALL_VERIFY_FLAGS;
            }
            if (this.getProgram().length > 10000 || scriptPubKey.getProgram().length > 10000) {
                throw new ScriptException("Script larger than 10,000 bytes");
            }
            LinkedList<byte[]> stack = new LinkedList<byte[]>();
            LinkedList<byte[]> p2shStack = null;
            Script.executeScript(txContainingThis, scriptSigIndex, this, stack, verifyFlags);
            if (verifyFlags.contains((Object)VerifyFlag.P2SH)) {
                p2shStack = new LinkedList<byte[]>(stack);
            }
            Script.executeScript(txContainingThis, scriptSigIndex, scriptPubKey, stack, verifyFlags);
            if (stack.size() == 0) {
                throw new ScriptException("Stack empty at end of script execution.");
            }
            if (!Script.castToBool(stack.pollLast())) {
                throw new ScriptException("Script resulted in a non-true stack: " + stack);
            }
            if (verifyFlags.contains((Object)VerifyFlag.P2SH) && scriptPubKey.isPayToScriptHash()) {
                for (ScriptChunk chunk : this.chunks) {
                    if (!chunk.isOpCode() || chunk.opcode <= 96) continue;
                    throw new ScriptException("Attempted to spend a P2SH scriptPubKey with a script that contained script ops");
                }
                byte[] scriptPubKeyBytes = p2shStack.pollLast();
                Script scriptPubKeyP2SH = new Script(scriptPubKeyBytes);
                Script.executeScript(txContainingThis, scriptSigIndex, scriptPubKeyP2SH, p2shStack, verifyFlags);
                if (p2shStack.size() == 0) {
                    throw new ScriptException("P2SH stack empty at end of script execution.");
                }
                if (!Script.castToBool(p2shStack.pollLast())) {
                    throw new ScriptException("P2SH script execution resulted in a non-true stack");
                }
            }
        }
        catch (ScriptException e) {
            return false;
        }
        return true;
    }

    private byte[] getQuickProgram() {
        if (this.program != null) {
            return this.program;
        }
        return this.getProgram();
    }

    public ScriptType getScriptType() {
        ScriptType type = ScriptType.NO_TYPE;
        if (this.isSentToAddress()) {
            type = ScriptType.P2PKH;
        } else if (this.isSentToRawPubKey()) {
            type = ScriptType.PUB_KEY;
        } else if (this.isPayToScriptHash()) {
            type = ScriptType.P2SH;
        }
        return type;
    }

    public static boolean verifySign(byte[] digestData, byte[] signData, byte[] publicKey) {
        return ECKey.verify((byte[])digestData, (byte[])signData, (byte[])publicKey);
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        return Arrays.equals(this.getQuickProgram(), ((Script)o).getQuickProgram());
    }

    public int hashCode() {
        return Arrays.hashCode(this.getQuickProgram());
    }

    public static enum VerifyFlag {
        P2SH,
        STRICTENC,
        DERSIG,
        LOW_S,
        NULLDUMMY,
        SIGPUSHONLY,
        MINIMALDATA,
        DISCOURAGE_UPGRADABLE_NOPS,
        CLEANSTACK,
        CHECKLOCKTIMEVERIFY;

    }

    public static enum ScriptType {
        NO_TYPE,
        P2PKH,
        PUB_KEY,
        P2SH;

    }
}

